'use client';

import React, { useState } from 'react';
import { BookOpen, Calendar, Clock, MapPin, Users, Zap, CheckCircle, Instagram, Link, Menu, X } from 'lucide-react';

// --- Data Definitions ---

const EVENT_DATA = {
  date: "Sat, 8th Nov, 2025",
  time: "5pm – 8pm GMT",
  venue: "Fully Virtual (Zoom & Google Meet)",
  cost: "Free to Attend",
  ctaLink: "https://www.eventbrite.com/e/tech-cohort-day-2025-tickets-1825846158009", // Anchor link to the registration section
  title: "Tech Cohort Day 2025",
  subtitle: "Breaking Into Tech: Pathways, Skills & Opportunities",
};

const BENEFITS = [
  { icon: Users, text: "Hear from 7 inspiring speakers across tech roles" },
  { icon: BookOpen, text: "Learn beginner-friendly skills in live demo workshops" },
  { icon: Zap, text: "Discover real-life career switch stories" },
  { icon: CheckCircle, text: "Get expert tips on portfolios, LinkedIn & remote work" },
  { icon: Link, text: "Connect with mentors and peers" },
  { icon: BookOpen, text: "Receive a curated Tech Starter Pack" },
  { icon: Clock, text: "Join our post-event community (A form for people to fill)" },
];

const SPEAKERS = [
  { name: "Adedayo Adesuyi", title: "Product Designer", imageUrl: "/Host/AdedayoAdesuyi.jpg" },
  { name: "Angelfortune Ogbeta", title: "Education advocate and entrepreneur", imageUrl: "/Host/AngelFortune.jpg" },
  { name: "Ayomikun Oloyede", title: "Software Engineer", imageUrl: "/Host/AyomikunOloyede.jpg" },
  { name: "Abisola Asubiojo", title: "Social Media Strategist", imageUrl: "/Host/AbisolsAsubiojo.jpg" },
  { name: "Damilola Akindoyin", title: "Brand Communication Strategist", imageUrl: "/Host/DamilolaA.jpg" },
  { name: "Solomon K. Yateghtegh", title: "Founder, Skyhub Nigeria", imageUrl: "/Host/SolomonK.jpg" },
  { name: "Harrison Ugwu", title: "Software Engineer (Co Founder x CTO Attesta Digital)", imageUrl: "/Host/HarrisonUgwu.jpg" },
];

// --- Utility Components ---

const CTAButton = ({ text, href, className = '' }) => (
  <a
    href={href}
    className={`inline-flex items-center justify-center px-8 py-3 text-base font-semibold leading-6 text-white bg-indigo-600 rounded-full shadow-lg hover:bg-indigo-700 transition duration-300 ease-in-out transform hover:scale-[1.02] ${className}`}
  >
    {text}
  </a>
);


const Header = () => (
  <header className="absolute top-0 left-0 right-0 z-10 p-4 lg:p-6 bg-transparent">
    <div className="flex justify-between items-center max-w-7xl mx-auto">
      <div className="text-xl font-extrabold text-white tracking-wider">
        TECHAM
      </div>
      <nav className="hidden md:flex space-x-6">
        {['Overview', 'About', 'Benefits', 'Speakers'].map((item) => (
          <a key={item} href={`#${item.toLowerCase().replace(' ', '')}`} className="text-white hover:text-indigo-400 transition">
            {item}
          </a>
        ))}
      </nav>
      <CTAButton text="Register Now" href={EVENT_DATA.ctaLink} className="hidden md:flex !py-2 !px-4 !text-sm" />
      <MobileMenu />
    </div>
  </header>
);

const MobileMenu = () => {
    const [isOpen, setIsOpen] = useState(false);
    return (
        <div className="md:hidden">
            <button onClick={() => setIsOpen(!isOpen)} className="text-white focus:outline-none p-2">
                {isOpen ? <X size={24} /> : <Menu size={24} />}
            </button>
            {isOpen && (
                <div className="absolute top-16 right-4 w-48 bg-gray-800 rounded-lg shadow-xl p-4 space-y-3 flex flex-col transition-all duration-300">
                    {['Overview', 'About', 'Benefits', 'Speakers'].map((item) => (
                        <a 
                            key={item} 
                            href={`#${item.toLowerCase().replace(' ', '')}`} 
                            onClick={() => setIsOpen(false)}
                            className="text-white hover:text-indigo-400 transition block py-2 border-b border-gray-700 last:border-b-0"
                        >
                            {item}
                        </a>
                    ))}
                    <CTAButton text="Register Now" href={EVENT_DATA.ctaLink} className="w-full !py-2 !text-sm" />
                </div>
            )}
        </div>
    );
};

const Hero = () => {
    // Note: The original image suggests a dark background with abstract shapes.
    // Using a gradient and abstract blobs to mimic the visual style.
    return (
      <section id="hero" className="relative h-screen flex items-center justify-center overflow-hidden bg-gray-900 text-white p-4">
        {/* Abstract shapes (mimicking the original design) */}
        <div className="absolute top-10 left-1/4 w-32 h-32 bg-green-400 rounded-full opacity-30 blur-2xl animate-pulse"></div>
        <div className="absolute bottom-10 right-1/4 w-40 h-40 bg-pink-500 rounded-full opacity-30 blur-2xl animate-pulse delay-500"></div>

        <div className="max-w-7xl mx-auto grid md:grid-cols-2 gap-10 items-center relative z-10">
          {/* Text Content */}
          <div className="space-y-6 md:space-y-8 text-center md:text-left pt-20 md:pt-0">
            <p className="text-indigo-400 font-semibold tracking-widest text-lg uppercase">{EVENT_DATA.title}</p>
            <h1 className="text-5xl sm:text-7xl font-extrabold leading-tight">
              BREAKING INTO TECH:
              <span className="block text-indigo-400 mt-2">Pathways, Skills & Opportunities</span>
            </h1>
            <p className="text-lg text-gray-300 max-w-lg mx-auto md:mx-0">
              Whether you’re a career-switcher, student, creative, or professional, this event is your gateway to discovering real pathways, in-demand skills, and global opportunities in the tech world.
            </p>
            <div className="pt-4">
                <CTAButton text="Register Now" href={EVENT_DATA.ctaLink} />
            </div>
          </div>

          {/* Event Details Card (Stylized as per image) */}
          <div className="hidden md:flex justify-end pt-20">
            <div className="bg-gray-800/80 backdrop-blur-sm p-6 rounded-xl shadow-2xl space-y-4 max-w-sm w-full">
              <h3 className="text-xl font-bold text-white mb-4 border-b border-indigo-500 pb-2">Event Details</h3>
              <div className="flex items-center space-x-3">
                <Calendar className="w-5 h-5 text-indigo-400" />
                <div>
                  <p className="text-sm text-gray-400">Date & Time:</p>
                  <p className="font-medium">{EVENT_DATA.date} / {EVENT_DATA.time}</p>
                </div>
              </div>
              <div className="flex items-center space-x-3">
                <MapPin className="w-5 h-5 text-indigo-400" />
                <div>
                  <p className="text-sm text-gray-400">Venue:</p>
                  <p className="font-medium">{EVENT_DATA.venue}</p>
                </div>
              </div>
              <div className="flex items-center space-x-3">
                <Zap className="w-5 h-5 text-indigo-400" />
                <div>
                  <p className="text-sm text-gray-400">Cost:</p>
                  <p className="font-medium">Free to Attend</p>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>
    );
};

const EventOverview = () => (
  <section id="overview" className="py-16 bg-white text-gray-800 px-4">
    <div className="max-w-5xl mx-auto text-center">
      <h2 className="text-4xl font-extrabold mb-2">{EVENT_DATA.title}</h2>
      <p className="text-2xl font-medium text-indigo-600">{EVENT_DATA.subtitle}</p>

      <div className="mt-10 grid grid-cols-2 md:grid-cols-4 gap-6">
        <OverviewCard icon={Calendar} label="Date" detail={EVENT_DATA.date} />
        <OverviewCard icon={Clock} label="Time" detail={EVENT_DATA.time} />
        <OverviewCard icon={MapPin} label="Venue" detail="Fully Virtual" subDetail="(Zoom & Google Meet)" />
        <OverviewCard icon={CheckCircle} label="Cost" detail="Free to Attend" />
      </div>
    </div>
  </section>
);

const OverviewCard = ({ icon: Icon, label, detail, subDetail }) => (
    <div className="p-6 bg-gray-50 rounded-xl shadow-md transition duration-300 hover:shadow-lg">
        <Icon className="w-8 h-8 text-indigo-600 mb-3 mx-auto" />
        <p className="text-sm font-semibold uppercase text-gray-500">{label}</p>
        <p className="text-xl font-bold text-gray-800 mt-1">{detail}</p>
        {subDetail && <p className="text-xs text-gray-500 mt-1">{subDetail}</p>}
    </div>
);


const AboutUs = () => (
  <section id="about" className="py-20 bg-gray-900 text-white px-4">
    <div className="max-w-5xl mx-auto grid md:grid-cols-2 gap-12 items-center">
      <div>
        <h2 className="text-xs font-semibold uppercase tracking-widest text-indigo-400 mb-2">Who We Are</h2>
        <h3 className="text-4xl font-extrabold mb-6">Tech Aspiring Minds Community</h3>
        <p className="text-xl font-light text-gray-300 mb-4">Empowering New Voices in Tech</p>
        <p className="text-lg text-gray-400">
          Tech Aspiring Minds is a growing community of career-switchers, students, creatives, and professionals exploring the world of tech. We believe tech should be inclusive, practical, and accessible — and we’re here to help you take your first step.
        </p>
      </div>
      <div className="hidden md:block">
        {/* Placeholder image/illustration */}
        <div className="w-full h-80 bg-indigo-600/20 rounded-xl flex items-center justify-center border-4 border-indigo-600 border-dashed">
            <Users className="w-16 h-16 text-indigo-400" />
        </div>
      </div>
    </div>
  </section>
);

const WhyJoin = () => (
  <section id="benefits" className="py-20 bg-white text-gray-800 px-4">
    <div className="max-w-5xl mx-auto">
      <h2 className="text-4xl font-extrabold text-center mb-12">Why Attend Tech Cohort Day?</h2>
      
      <div className="grid md:grid-cols-2 gap-8">
        {BENEFITS.map((benefit, index) => (
          <div key={index} className="flex items-start space-x-4 p-5 bg-gray-50 rounded-lg shadow-md transition duration-300 hover:shadow-xl hover:bg-indigo-50">
            <div className="flex-shrink-0">
              <benefit.icon className="w-6 h-6 text-indigo-600 mt-1" />
            </div>
            <p className="text-lg font-medium text-gray-700">{benefit.text}</p>
          </div>
        ))}
      </div>
    </div>
  </section>
);

const Speakers = () => (
  <section id="speakers" className="py-20 bg-gray-800 text-white px-4">
    <div className="max-w-7xl mx-auto">
      <h2 className="text-4xl font-extrabold text-center mb-4">Meet the Speakers</h2>
      <p className="text-center text-gray-400 mb-12 text-xl">Learn from the best minds in Product, Engineering, Marketing, and Education.</p>

      <div className="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-4 gap-8">
        {SPEAKERS.map((speaker, index) => (
          <SpeakerCard key={index} speaker={speaker} />
        ))}
      </div>
    </div>
  </section>
);

const SpeakerCard = ({ speaker }) => (
  <div className="flex flex-col items-center text-center p-6 bg-gray-900 rounded-xl shadow-lg transition duration-300 hover:bg-gray-700">
    <img
      src={speaker.imageUrl}
      alt={speaker.name}
      className="w-24 h-24 rounded-full object-cover mb-4 ring-4 ring-indigo-500"
      // Fallback for failed image load
      onError={(e) => { e.target.onerror = null; e.target.src = "https://placehold.co/100x100/3b82f6/ffffff?text=U" }}
    />
    <h4 className="text-xl font-bold mb-1">{speaker.name}</h4>
    <p className="text-sm text-indigo-400 font-medium">{speaker.title}</p>
  </div>
);

const RegistrationCTA = () => (
  <section id="registration" className="py-20 bg-white text-gray-800 px-4">
    <div className="max-w-3xl mx-auto text-center p-8 border-4 border-indigo-600 border-dashed rounded-xl shadow-xl">
      <h2 className="text-5xl font-extrabold mb-4">Ready to Join?</h2>
      <p className="text-lg text-gray-600 mb-8">
        Discover your pathway into tech at the **Tech Cohort Day 2025** event.
      </p>
      
      <div className="space-y-4 mb-8">
        <p className="text-xl font-semibold text-green-600 flex items-center justify-center">
          <CheckCircle className="w-6 h-6 mr-2" /> It’s free and open to all
        </p>
        <p className="text-xl font-semibold text-indigo-600 flex items-center justify-center">
          <Link className="w-6 h-6 mr-2" /> Click below to secure your spot!
        </p>
      </div>

      <CTAButton text="Secure My Free Spot Now" href="https://www.eventbrite.com/e/tech-cohort-day-2025-tickets-1825846158009" className="!text-xl" />

      <p className="text-sm text-gray-500 mt-6">
        *If using a QR code, you would display it here, linking to the registration page.
      </p>
    </div>
  </section>
);

const Footer = () => (
  <footer className="py-12 bg-gray-900 text-white px-4">
    <div className="max-w-7xl mx-auto flex flex-col sm:flex-row justify-between items-center text-center sm:text-left space-y-6 sm:space-y-0">
      
      {/* Community Info */}
      <div className="space-y-2">
        <h4 className="text-xl font-bold text-indigo-400">Tech Aspiring Minds</h4>
        <p className="text-sm text-gray-400">Empowering New Voices in Tech</p>
        <p className="text-sm text-gray-500">© 2025 Tech Aspiring Minds Community.</p>
      </div>

      {/* Stay Connected */}
      <div className="space-y-3">
        <h4 className="text-lg font-semibold border-b border-indigo-600 inline-block pb-1">Follow Us</h4>
        <div className="flex flex-col items-center sm:items-start space-y-2">
          <a href="https://instagram.com/techamcommunity" target="_blank" rel="noopener noreferrer" className="flex items-center text-gray-300 hover:text-indigo-400 transition">
            <Instagram className="w-5 h-5 mr-2" />
            @techamcommunity
          </a>
          <div className="text-sm text-gray-400 font-mono">
            #TechCohortDay2025 #TechAspiringMinds
          </div>
        </div>
      </div>
    </div>
  </footer>
);


// --- Main App Component ---
const App = () => {
  // Use 'Inter' font via Tailwind config (assumed to be available)

  return (
    <div className="min-h-screen bg-white font-sans antialiased">
      <Header />
      <main>
        <Hero />
        <EventOverview />
        <AboutUs />
        <WhyJoin />
        <Speakers />
        <RegistrationCTA />
      </main>
      <Footer />
    </div>
  );
};

export default App;
